/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.util;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.StorageSize;
import org.apache.hadoop.hbase.StorageUnit;
import org.apache.hadoop.hbase.regionserver.MemStoreLAB;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class MemorySizeUtil {
    public static final String MEMSTORE_SIZE_KEY = "hbase.regionserver.global.memstore.size";
    public static final String MEMSTORE_SIZE_OLD_KEY = "hbase.regionserver.global.memstore.upperLimit";
    public static final String MEMSTORE_SIZE_LOWER_LIMIT_KEY = "hbase.regionserver.global.memstore.size.lower.limit";
    public static final String MEMSTORE_SIZE_LOWER_LIMIT_OLD_KEY = "hbase.regionserver.global.memstore.lowerLimit";
    public static final String OFFHEAP_MEMSTORE_SIZE_KEY = "hbase.regionserver.offheap.global.memstore.size";
    public static final float DEFAULT_MEMSTORE_SIZE = 0.4f;
    public static final float DEFAULT_MEMSTORE_SIZE_LOWER_LIMIT = 0.95f;
    public static final String HBASE_REGION_SERVER_FREE_HEAP_MIN_MEMORY_SIZE_KEY = "hbase.regionserver.free.heap.min.memory.size";
    private static final Logger LOG = LoggerFactory.getLogger(MemorySizeUtil.class);
    private static final String JVM_HEAP_EXCEPTION = "Got an exception while attempting to read information about the JVM heap. Please submit this log information in a bug report and include your JVM settings, specifically the GC in use and any -XX options. Consider restarting the service.";

    public static MemoryUsage safeGetHeapMemoryUsage() {
        MemoryUsage usage = null;
        try {
            usage = ManagementFactory.getMemoryMXBean().getHeapMemoryUsage();
        }
        catch (RuntimeException exception) {
            LOG.warn(JVM_HEAP_EXCEPTION, (Throwable)exception);
        }
        return usage;
    }

    public static void validateRegionServerHeapMemoryAllocation(Configuration conf) {
        float minFreeHeapFraction;
        int minFreeHeapPercent;
        int maxAllowedUsed;
        float blockCacheFraction;
        int blockCachePercent;
        float memStoreFraction;
        int memStorePercent;
        int usedPercent;
        if (conf.get(MEMSTORE_SIZE_OLD_KEY) != null) {
            LOG.warn("hbase.regionserver.global.memstore.upperLimit is deprecated by hbase.regionserver.global.memstore.size");
        }
        if ((usedPercent = (memStorePercent = (int)((memStoreFraction = MemorySizeUtil.getGlobalMemStoreHeapPercent(conf, false)) * 100.0f)) + (blockCachePercent = (int)((blockCacheFraction = MemorySizeUtil.getBlockCacheHeapPercent(conf)) * 100.0f))) > (maxAllowedUsed = 100 - (minFreeHeapPercent = (int)((minFreeHeapFraction = MemorySizeUtil.getRegionServerMinFreeHeapFraction(conf)) * 100.0f)))) {
            throw new RuntimeException(String.format("RegionServer heap memory allocation is invalid: total memory usage exceeds 100%% (memStore + blockCache + requiredFreeHeap). Check the following configuration values:%n  - %s = %.2f%n  - %s = %s%n  - %s = %s%n  - %s = %s", MEMSTORE_SIZE_KEY, Float.valueOf(memStoreFraction), "hfile.block.cache.memory.size", conf.get("hfile.block.cache.memory.size"), "hfile.block.cache.size", conf.get("hfile.block.cache.size"), HBASE_REGION_SERVER_FREE_HEAP_MIN_MEMORY_SIZE_KEY, conf.get(HBASE_REGION_SERVER_FREE_HEAP_MIN_MEMORY_SIZE_KEY)));
        }
    }

    private static long getRegionServerMinFreeHeapInBytes(Configuration conf) {
        String key = HBASE_REGION_SERVER_FREE_HEAP_MIN_MEMORY_SIZE_KEY;
        try {
            return Long.parseLong(conf.get(HBASE_REGION_SERVER_FREE_HEAP_MIN_MEMORY_SIZE_KEY));
        }
        catch (NumberFormatException e) {
            return (long)StorageSize.getStorageSize(conf.get(HBASE_REGION_SERVER_FREE_HEAP_MIN_MEMORY_SIZE_KEY), -1.0, StorageUnit.BYTES);
        }
    }

    public static float getRegionServerMinFreeHeapFraction(Configuration conf) {
        MemoryUsage usage = MemorySizeUtil.safeGetHeapMemoryUsage();
        if (usage == null) {
            return 0.0f;
        }
        long minFreeHeapInBytes = MemorySizeUtil.getRegionServerMinFreeHeapInBytes(conf);
        if (minFreeHeapInBytes >= 0L) {
            return (float)minFreeHeapInBytes / (float)usage.getMax();
        }
        return 0.2f;
    }

    public static float getGlobalMemStoreHeapPercent(Configuration c, boolean logInvalid) {
        float limit = c.getFloat(MEMSTORE_SIZE_KEY, c.getFloat(MEMSTORE_SIZE_OLD_KEY, 0.4f));
        if (limit > 0.8f || limit <= 0.0f) {
            if (logInvalid) {
                LOG.warn("Setting global memstore limit to default of 0.4 because supplied value outside allowed range of (0 -> 0.8]");
            }
            limit = 0.4f;
        }
        return limit;
    }

    public static float getGlobalMemStoreHeapLowerMark(Configuration conf, boolean honorOldConfig) {
        String lowMarkPercentStr = conf.get(MEMSTORE_SIZE_LOWER_LIMIT_KEY);
        if (lowMarkPercentStr != null) {
            float lowMarkPercent = Float.parseFloat(lowMarkPercentStr);
            if (lowMarkPercent > 1.0f) {
                LOG.error("Bad configuration value for hbase.regionserver.global.memstore.size.lower.limit: " + lowMarkPercent + ". Using 1.0f instead.");
                lowMarkPercent = 1.0f;
            }
            return lowMarkPercent;
        }
        if (!honorOldConfig) {
            return 0.95f;
        }
        String lowerWaterMarkOldValStr = conf.get(MEMSTORE_SIZE_LOWER_LIMIT_OLD_KEY);
        if (lowerWaterMarkOldValStr != null) {
            LOG.warn("hbase.regionserver.global.memstore.lowerLimit is deprecated. Instead use hbase.regionserver.global.memstore.size.lower.limit");
            float lowerWaterMarkOldVal = Float.parseFloat(lowerWaterMarkOldValStr);
            float upperMarkPercent = MemorySizeUtil.getGlobalMemStoreHeapPercent(conf, false);
            if (lowerWaterMarkOldVal > upperMarkPercent) {
                lowerWaterMarkOldVal = upperMarkPercent;
                LOG.error("Value of hbase.regionserver.global.memstore.lowerLimit (" + lowerWaterMarkOldVal + ") is greater than global memstore limit (" + upperMarkPercent + ") set by " + MEMSTORE_SIZE_KEY + "/" + MEMSTORE_SIZE_OLD_KEY + ". Setting memstore lower limit to " + upperMarkPercent);
            }
            return lowerWaterMarkOldVal / upperMarkPercent;
        }
        return 0.95f;
    }

    public static Pair<Long, MemoryType> getGlobalMemStoreSize(Configuration conf) {
        long offheapMSGlobal = conf.getLong(OFFHEAP_MEMSTORE_SIZE_KEY, 0L);
        if (offheapMSGlobal > 0L) {
            if (MemStoreLAB.isEnabled(conf)) {
                long globalMemStoreLimit = offheapMSGlobal * 1024L * 1024L;
                return new Pair<Long, MemoryType>(globalMemStoreLimit, MemoryType.NON_HEAP);
            }
            LOG.warn("There is no relevance of configuring 'hbase.regionserver.offheap.global.memstore.size' when 'hbase.hregion.memstore.mslab.enabled' is turned off. Going with on heap global memstore size ('hbase.regionserver.global.memstore.size')");
        }
        return new Pair<Long, MemoryType>(MemorySizeUtil.getOnheapGlobalMemStoreSize(conf), MemoryType.HEAP);
    }

    public static long getOnheapGlobalMemStoreSize(Configuration conf) {
        long max = -1L;
        MemoryUsage usage = MemorySizeUtil.safeGetHeapMemoryUsage();
        if (usage != null) {
            max = usage.getMax();
        }
        float globalMemStorePercent = MemorySizeUtil.getGlobalMemStoreHeapPercent(conf, true);
        return (long)((float)max * globalMemStorePercent);
    }

    public static float getBlockCacheHeapPercent(Configuration conf) {
        long l1CacheSizeInBytes = MemorySizeUtil.getBlockCacheSizeInBytes(conf);
        if (l1CacheSizeInBytes > 0L) {
            MemoryUsage usage = MemorySizeUtil.safeGetHeapMemoryUsage();
            return usage == null ? 0.0f : (float)l1CacheSizeInBytes / (float)usage.getMax();
        }
        return conf.getFloat("hfile.block.cache.size", 0.4f);
    }

    public static long getBlockCacheSizeInBytes(Configuration conf) {
        String key = "hfile.block.cache.memory.size";
        try {
            return Long.parseLong(conf.get("hfile.block.cache.memory.size"));
        }
        catch (NumberFormatException e) {
            return (long)StorageSize.getStorageSize(conf.get("hfile.block.cache.memory.size"), -1.0, StorageUnit.BYTES);
        }
    }

    public static long getOnHeapCacheSize(Configuration conf) {
        float cachePercentage = MemorySizeUtil.getBlockCacheHeapPercent(conf);
        if (cachePercentage <= 1.0E-4f) {
            return -1L;
        }
        if ((double)cachePercentage > 1.0) {
            throw new IllegalArgumentException("hfile.block.cache.size must be between 0.0 and 1.0, and not > 1.0");
        }
        MemoryUsage usage = MemorySizeUtil.safeGetHeapMemoryUsage();
        if (usage == null) {
            return -1L;
        }
        long heapMax = usage.getMax();
        float onHeapCacheFixedSize = (float)conf.getLong("hfile.onheap.block.cache.fixed.size", 0L) / (float)heapMax;
        if (onHeapCacheFixedSize > 0.0f && onHeapCacheFixedSize < cachePercentage) {
            return (long)((float)heapMax * onHeapCacheFixedSize);
        }
        long cacheSizeInBytes = MemorySizeUtil.getBlockCacheSizeInBytes(conf);
        return cacheSizeInBytes > 0L ? cacheSizeInBytes : (long)((float)heapMax * cachePercentage);
    }

    public static long getBucketCacheSize(Configuration conf) {
        float bucketCacheSize = conf.getFloat("hbase.bucketcache.size", 0.0f);
        if (bucketCacheSize < 1.0f) {
            throw new IllegalArgumentException("Bucket Cache should be minimum 1 MB in size.Configure 'hbase.bucketcache.size' with > 1 value");
        }
        return (long)(bucketCacheSize * 1024.0f * 1024.0f);
    }
}

