/*
 * Decompiled with CFR 0.152.
 */
package org.apache.paimon.table.source.snapshot;

import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Optional;
import java.util.TimeZone;
import javax.annotation.Nullable;
import org.apache.paimon.CoreOptions;
import org.apache.paimon.Snapshot;
import org.apache.paimon.options.Options;
import org.apache.paimon.schema.SchemaManager;
import org.apache.paimon.schema.TableSchema;
import org.apache.paimon.table.FileStoreTable;
import org.apache.paimon.table.source.snapshot.StaticFromSnapshotStartingScanner;
import org.apache.paimon.table.source.snapshot.StaticFromTagStartingScanner;
import org.apache.paimon.table.source.snapshot.StaticFromTimestampStartingScanner;
import org.apache.paimon.table.source.snapshot.StaticFromWatermarkStartingScanner;
import org.apache.paimon.utils.ChangelogManager;
import org.apache.paimon.utils.DateTimeUtils;
import org.apache.paimon.utils.FunctionWithException;
import org.apache.paimon.utils.Preconditions;
import org.apache.paimon.utils.SnapshotManager;
import org.apache.paimon.utils.TagManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimeTravelUtil {
    private static final Logger LOG = LoggerFactory.getLogger(TimeTravelUtil.class);
    private static final String WATERMARK_PREFIX = "watermark-";
    private static final String[] SCAN_KEYS = new String[]{CoreOptions.SCAN_SNAPSHOT_ID.key(), CoreOptions.SCAN_TAG_NAME.key(), CoreOptions.SCAN_WATERMARK.key(), CoreOptions.SCAN_TIMESTAMP.key(), CoreOptions.SCAN_TIMESTAMP_MILLIS.key()};

    public static Snapshot tryTravelOrLatest(FileStoreTable table) {
        return TimeTravelUtil.tryTravelToSnapshot(table).orElseGet(() -> table.latestSnapshot().orElse(null));
    }

    public static Optional<Snapshot> tryTravelToSnapshot(FileStoreTable table) {
        return TimeTravelUtil.tryTravelToSnapshot(table.coreOptions().toConfiguration(), table.snapshotManager(), table.tagManager());
    }

    public static Optional<Snapshot> tryTravelToSnapshot(Options options, SnapshotManager snapshotManager, TagManager tagManager) {
        Snapshot snapshot;
        TimeTravelUtil.adaptScanVersion(options, tagManager);
        ArrayList<String> scanHandleKey = new ArrayList<String>(1);
        for (String key : SCAN_KEYS) {
            if (!options.containsKey(key)) continue;
            scanHandleKey.add(key);
        }
        if (scanHandleKey.isEmpty()) {
            return Optional.empty();
        }
        Preconditions.checkArgument(scanHandleKey.size() == 1, String.format("Only one of the following parameters may be set : %s", Arrays.toString(SCAN_KEYS)));
        String key = (String)scanHandleKey.get(0);
        CoreOptions coreOptions = new CoreOptions(options);
        if (key.equals(CoreOptions.SCAN_SNAPSHOT_ID.key())) {
            snapshot = new StaticFromSnapshotStartingScanner(snapshotManager, coreOptions.scanSnapshotId()).getSnapshot();
        } else if (key.equals(CoreOptions.SCAN_WATERMARK.key())) {
            snapshot = new StaticFromWatermarkStartingScanner(snapshotManager, coreOptions.scanWatermark()).getSnapshot();
        } else if (key.equals(CoreOptions.SCAN_TIMESTAMP.key())) {
            snapshot = new StaticFromTimestampStartingScanner(snapshotManager, DateTimeUtils.parseTimestampData(coreOptions.scanTimestamp(), 3, TimeZone.getDefault()).getMillisecond()).getSnapshot();
        } else if (key.equals(CoreOptions.SCAN_TIMESTAMP_MILLIS.key())) {
            snapshot = new StaticFromTimestampStartingScanner(snapshotManager, coreOptions.scanTimestampMills()).getSnapshot();
        } else if (key.equals(CoreOptions.SCAN_TAG_NAME.key())) {
            snapshot = new StaticFromTagStartingScanner(snapshotManager, coreOptions.scanTagName()).getSnapshot();
        } else {
            throw new UnsupportedOperationException("Unsupported time travel mode: " + key);
        }
        return Optional.of(snapshot);
    }

    private static void adaptScanVersion(Options options, TagManager tagManager) {
        String version = options.remove(CoreOptions.SCAN_VERSION.key());
        if (version == null) {
            return;
        }
        if (tagManager.tagExists(version)) {
            options.set(CoreOptions.SCAN_TAG_NAME, version);
        } else if (version.startsWith(WATERMARK_PREFIX)) {
            long watermark = Long.parseLong(version.substring(WATERMARK_PREFIX.length()));
            options.set(CoreOptions.SCAN_WATERMARK, watermark);
        } else if (version.chars().allMatch(Character::isDigit)) {
            options.set(CoreOptions.SCAN_SNAPSHOT_ID.key(), version);
        } else {
            options.set(CoreOptions.SCAN_TAG_NAME.key(), version);
            throw new RuntimeException("Cannot find a time travel version for " + version);
        }
    }

    @Nullable
    public static Long earlierThanTimeMills(SnapshotManager snapshotManager, ChangelogManager changelogManager, long timestampMills, boolean startFromChangelog, boolean returnNullIfTooEarly) {
        Long latest = snapshotManager.latestSnapshotId();
        if (latest == null) {
            return null;
        }
        Snapshot earliestSnapshot = TimeTravelUtil.earliestSnapshot(snapshotManager, changelogManager, startFromChangelog, latest);
        if (earliestSnapshot == null) {
            return latest - 1L;
        }
        if (earliestSnapshot.timeMillis() >= timestampMills) {
            return returnNullIfTooEarly ? null : Long.valueOf(earliestSnapshot.id() - 1L);
        }
        long earliest = earliestSnapshot.id();
        while (earliest < latest) {
            Snapshot snapshot;
            long mid = (earliest + latest + 1L) / 2L;
            Snapshot snapshot2 = snapshot = startFromChangelog ? TimeTravelUtil.changelogOrSnapshot(snapshotManager, changelogManager, mid) : snapshotManager.snapshot(mid);
            if (snapshot.timeMillis() < timestampMills) {
                earliest = mid;
                continue;
            }
            latest = mid - 1L;
        }
        return earliest;
    }

    @Nullable
    private static Snapshot earliestSnapshot(SnapshotManager snapshotManager, ChangelogManager changelogManager, boolean includeChangelog, @Nullable Long stopSnapshotId) {
        Long snapshotId = null;
        if (includeChangelog) {
            snapshotId = changelogManager.earliestLongLivedChangelogId();
        }
        if (snapshotId == null) {
            snapshotId = snapshotManager.earliestSnapshotId();
        }
        if (snapshotId == null) {
            return null;
        }
        if (stopSnapshotId == null) {
            stopSnapshotId = snapshotId + 3L;
        }
        FunctionWithException snapshotFunction = includeChangelog ? s -> TimeTravelUtil.tryGetChangelogOrSnapshot(snapshotManager, changelogManager, s) : snapshotManager::tryGetSnapshot;
        while (true) {
            try {
                return snapshotFunction.apply(snapshotId);
            }
            catch (FileNotFoundException e) {
                Long l = snapshotId;
                Long l2 = snapshotId = Long.valueOf(snapshotId + 1L);
                if (snapshotId > stopSnapshotId) {
                    return null;
                }
                LOG.warn("The earliest snapshot or changelog was once identified but disappeared. It might have been expired by other jobs operating on this table. Searching for the second earliest snapshot or changelog instead. ");
                continue;
            }
            break;
        }
    }

    private static Snapshot tryGetChangelogOrSnapshot(SnapshotManager snapshotManager, ChangelogManager changelogManager, long snapshotId) throws FileNotFoundException {
        if (changelogManager.longLivedChangelogExists(snapshotId)) {
            return changelogManager.tryGetChangelog(snapshotId);
        }
        return snapshotManager.tryGetSnapshot(snapshotId);
    }

    private static Snapshot changelogOrSnapshot(SnapshotManager snapshotManager, ChangelogManager changelogManager, long snapshotId) {
        if (changelogManager.longLivedChangelogExists(snapshotId)) {
            return changelogManager.changelog(snapshotId);
        }
        return snapshotManager.snapshot(snapshotId);
    }

    public static void checkRescaleBucketForIncrementalDiffQuery(SchemaManager schemaManager, Snapshot start, Snapshot end) {
        int endBucketNumber;
        int startBucketNumber;
        if (start.schemaId() != end.schemaId() && (startBucketNumber = TimeTravelUtil.bucketNumber(schemaManager, start.schemaId())) != (endBucketNumber = TimeTravelUtil.bucketNumber(schemaManager, end.schemaId()))) {
            throw new InconsistentTagBucketException(start.id(), end.id(), String.format("The bucket number of two snapshots are different (%s, %s), which is not supported in incremental diff query.", startBucketNumber, endBucketNumber));
        }
    }

    private static int bucketNumber(SchemaManager schemaManager, long schemaId) {
        TableSchema schema = schemaManager.schema(schemaId);
        return CoreOptions.fromMap(schema.options()).bucket();
    }

    @Nullable
    public static Long scanTimestampMills(Options options) {
        String timestampStr = options.get(CoreOptions.SCAN_TIMESTAMP);
        Long timestampMillis = options.get(CoreOptions.SCAN_TIMESTAMP_MILLIS);
        if (timestampMillis == null && timestampStr != null) {
            return DateTimeUtils.parseTimestampData(timestampStr, 3, TimeZone.getDefault()).getMillisecond();
        }
        return timestampMillis;
    }

    public static class InconsistentTagBucketException
    extends RuntimeException {
        private final long startSnapshotId;
        private final long endSnapshotId;

        public InconsistentTagBucketException(long startSnapshotId, long endSnapshotId, String message) {
            super(message);
            this.startSnapshotId = startSnapshotId;
            this.endSnapshotId = endSnapshotId;
        }

        public long startSnapshotId() {
            return this.startSnapshotId;
        }

        public long endSnapshotId() {
            return this.endSnapshotId;
        }
    }
}

