/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 *   SPDX-License-Identifier: Apache-2.0
 */

package org.apache.jena.shacl.engine.constraint;

import static org.apache.jena.shacl.compact.writer.CompactOut.compact;
import static org.apache.jena.shacl.lib.ShLib.displayStr;

import java.util.Objects;
import java.util.Set;

import org.apache.jena.atlas.io.IndentedWriter;
import org.apache.jena.graph.Node;
import org.apache.jena.riot.out.NodeFormatter;
import org.apache.jena.shacl.engine.ValidationContext;
import org.apache.jena.shacl.parser.ConstraintVisitor;
import org.apache.jena.shacl.parser.Shape;
import org.apache.jena.shacl.vocabulary.SHACL;
import org.apache.jena.sparql.expr.Expr;
import org.apache.jena.sparql.path.Path;

/** sh:lessThanOrEquals */
public class LessThanOrEqualsConstraint extends ConstraintPairwise {

    public LessThanOrEqualsConstraint(Node value) {
        super(value, SHACL.LessThanOrEqualsConstraintComponent);
    }

    @Override
    public void validate(ValidationContext vCxt, Shape shape, Node focusNode, Path path,
                         Set<Node> pathNodes, Set<Node> compareNodes) {
        for ( Node vn : pathNodes ) {
            for ( Node v : compareNodes ) {
                int r = super.compare(vn, v) ;
                boolean passed = true;
                if ( r != Expr.CMP_LESS && r != Expr.CMP_EQUAL ) {
                    passed = false;
                    String msg = toString()+": value node "+displayStr(vn)+" is not less than or equal to "+displayStr(v);
                    vCxt.reportEntry(msg, shape, focusNode, path, vn, this);
                }
                final boolean finalPassed = passed;
                    vCxt.notifyValidationListener(() ->
                                    makeEventSingleCompareNode(vCxt, shape,
                                                    focusNode, path, vn, v,
                                                    finalPassed));
            }
        }
    }

    @Override
    public void visit(ConstraintVisitor visitor){
        visitor.visit(this);
    }

    @Override
    public void printCompact(IndentedWriter out, NodeFormatter nodeFmt) {
        compact(out, nodeFmt, "lessThanOrEquals", value);
    }

    @Override
    public String toString() {
        return "LessThanOrEquals["+displayStr(value)+"]";
    }

    @Override
    public int hashCode() {
        return Objects.hash(value, 4);
    }
}
