/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.plugins.assembly.archive.archiver;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.codehaus.plexus.archiver.Archiver;
import org.codehaus.plexus.archiver.ArchiverException;
import org.codehaus.plexus.archiver.FileSet;
import org.codehaus.plexus.archiver.diags.TrackingArchiver;
import org.codehaus.plexus.archiver.jar.JarArchiver;
import org.codehaus.plexus.archiver.util.DefaultFileSet;
import org.codehaus.plexus.components.io.fileselectors.FileInfo;
import org.codehaus.plexus.components.io.fileselectors.FileSelector;
import org.codehaus.plexus.components.io.functions.InputStreamTransformer;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.Timeout;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.io.TempDir;
import org.mockito.ArgumentCaptor;
import org.mockito.junit.jupiter.MockitoExtension;
import org.mockito.junit.jupiter.MockitoSettings;
import org.mockito.quality.Strictness;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@MockitoSettings(strictness = Strictness.WARN)
@ExtendWith(MockitoExtension.class)
public class AssemblyProxyArchiverTest {
    @TempDir
    private File temporaryFolder;

    @Test
    @Timeout(value = 5000, unit = TimeUnit.MILLISECONDS)
    public void addFileSetSkipWhenSourceIsAssemblyWorkDir() throws IOException, ArchiverException {
        final File sources = temporaryFolder;

        final File workdir = new File(sources, "workdir");

        final TrackingArchiver tracker = new TrackingArchiver();
        final AssemblyProxyArchiver archiver = new AssemblyProxyArchiver("", tracker, null, null, null, workdir);

        archiver.setForced(true);

        final DefaultFileSet fs = new DefaultFileSet();
        fs.setDirectory(workdir);

        archiver.addFileSet(fs);

        assertTrue(tracker.added.isEmpty());
    }

    @Test
    @Timeout(value = 5000, unit = TimeUnit.MILLISECONDS)
    public void addFileSetAddExcludeWhenSourceContainsAssemblyWorkDir() throws IOException, ArchiverException {
        final File sources = temporaryFolder;

        final File workdir = new File(sources, "workdir");
        workdir.mkdir();

        Files.write(
                sources.toPath().resolve("test-included.txt"),
                Collections.singletonList("This is included"),
                StandardCharsets.UTF_8);
        Files.write(
                workdir.toPath().resolve("test-excluded.txt"),
                Collections.singletonList("This is excluded"),
                StandardCharsets.UTF_8);

        final TrackingArchiver tracker = new TrackingArchiver();
        final AssemblyProxyArchiver archiver = new AssemblyProxyArchiver("", tracker, null, null, null, workdir);

        archiver.setForced(true);

        final DefaultFileSet fs = new DefaultFileSet();
        fs.setDirectory(sources);

        archiver.addFileSet(fs);

        assertEquals(1, tracker.added.size());

        final TrackingArchiver.Addition addition = tracker.added.get(0);
        assertNotNull(addition.excludes);
        assertEquals(1, addition.excludes.length);
        assertEquals(workdir.getName(), addition.excludes[0]);
    }

    @Test
    public void addFileNoPermsCallAcceptFilesOnlyOnce() throws IOException, ArchiverException {
        final Archiver delegate = mock(Archiver.class);

        final CounterSelector counter = new CounterSelector(true);
        final List<FileSelector> selectors = new ArrayList<>();
        selectors.add(counter);

        final AssemblyProxyArchiver archiver =
                new AssemblyProxyArchiver("", delegate, null, selectors, null, new File("."));
        archiver.setForced(true);

        final File inputFile = File.createTempFile("junit", null, temporaryFolder);
        archiver.addFile(inputFile, "file.txt");

        assertEquals(1, counter.getCount());
        verify(delegate).addFile(inputFile, "file.txt");
        verify(delegate).setForced(true);
    }

    @Test
    @SuppressWarnings("deprecation")
    public void addDirectoryNoPermsCallAcceptFilesOnlyOnce() throws IOException, ArchiverException {
        final Archiver delegate = new JarArchiver();

        final File output = File.createTempFile("junit", null, temporaryFolder);

        delegate.setDestFile(output);

        final CounterSelector counter = new CounterSelector(true);
        final List<FileSelector> selectors = new ArrayList<>();
        selectors.add(counter);

        final AssemblyProxyArchiver archiver =
                new AssemblyProxyArchiver("", delegate, null, selectors, null, new File("."));

        archiver.setForced(true);

        final File dir = newFolder(temporaryFolder, "junit");
        Files.write(
                dir.toPath().resolve("file.txt"), Collections.singletonList("This is a test."), StandardCharsets.UTF_8);

        archiver.addDirectory(dir);

        archiver.createArchive();

        assertEquals(1, counter.getCount());
    }

    @Test
    public void assemblyWorkDir() {
        final Archiver delegate = mock(Archiver.class);
        final List<FileSelector> selectors = new ArrayList<>();

        final AssemblyProxyArchiver archiver = new AssemblyProxyArchiver(
                "prefix", delegate, null, selectors, null, new File(temporaryFolder, "module1"));

        FileSet fileSet = mock(FileSet.class);
        when(fileSet.getDirectory()).thenReturn(temporaryFolder);
        when(fileSet.getStreamTransformer()).thenReturn(mock(InputStreamTransformer.class));

        archiver.addFileSet(fileSet);

        ArgumentCaptor<FileSet> delFileSet = ArgumentCaptor.forClass(FileSet.class);
        verify(delegate).addFileSet(delFileSet.capture());

        assertThat(delFileSet.getValue().getDirectory(), is(fileSet.getDirectory()));
        assertThat(delFileSet.getValue().getExcludes(), is(new String[] {"module1"}));
        assertThat(delFileSet.getValue().getFileMappers(), is(fileSet.getFileMappers()));
        assertThat(delFileSet.getValue().getFileSelectors(), is(fileSet.getFileSelectors()));
        assertThat(delFileSet.getValue().getIncludes(), is(new String[0]));
        assertThat(delFileSet.getValue().getPrefix(), is("prefix/"));
        assertThat(delFileSet.getValue().getStreamTransformer(), is(fileSet.getStreamTransformer()));
    }

    private static final class CounterSelector implements FileSelector {

        private int count = 0;

        private boolean answer = false;

        CounterSelector(final boolean answer) {
            this.answer = answer;
        }

        int getCount() {
            return count;
        }

        public boolean isSelected(final FileInfo fileInfo) throws IOException {
            if (fileInfo.isFile()) {
                count++;
                System.out.println("Counting file: " + fileInfo.getName() + ". Current count: " + count);
            }

            return answer;
        }
    }

    private static File newFolder(File root, String... subDirs) throws IOException {
        String subFolder = String.join("/", subDirs);
        File result = new File(root, subFolder);
        if (!result.mkdirs()) {
            throw new IOException("Couldn't create folders " + root);
        }
        return result;
    }
}
