/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.apache.hadoop.hdds.scm.node.SCMNodeStorageStatMXBean;
import org.apache.hadoop.hdds.scm.node.StorageReportResult;
import org.apache.hadoop.ozone.container.common.impl.StorageLocationReport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SCMNodeStorageStatMap
implements SCMNodeStorageStatMXBean {
    static final Logger LOG = LoggerFactory.getLogger(SCMNodeStorageStatMap.class);
    private final double warningUtilizationThreshold;
    private final double criticalUtilizationThreshold;
    private final Map<UUID, Set<StorageLocationReport>> scmNodeStorageReportMap = new ConcurrentHashMap<UUID, Set<StorageLocationReport>>();

    public SCMNodeStorageStatMap(OzoneConfiguration conf) {
        this.warningUtilizationThreshold = conf.getDouble("hdds.datanode.storage.utilization.warning.threshold", 0.75);
        this.criticalUtilizationThreshold = conf.getDouble("hdds.datanode.storage.utilization.critical.threshold", 0.95);
    }

    public boolean isKnownDatanode(UUID datanodeID) {
        Preconditions.checkNotNull((Object)datanodeID);
        return this.scmNodeStorageReportMap.containsKey(datanodeID);
    }

    public List<UUID> getDatanodeList(UtilizationThreshold threshold) {
        return this.scmNodeStorageReportMap.entrySet().stream().filter(entry -> this.isThresholdReached(threshold, this.getScmUsedratio(this.getUsedSpace((UUID)entry.getKey()), this.getCapacity((UUID)entry.getKey())))).map(Map.Entry::getKey).collect(Collectors.toList());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void insertNewDatanode(UUID datanodeID, Set<StorageLocationReport> report) throws SCMException {
        Preconditions.checkNotNull(report);
        Preconditions.checkState((!report.isEmpty() ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)datanodeID);
        Map<UUID, Set<StorageLocationReport>> map = this.scmNodeStorageReportMap;
        synchronized (map) {
            if (this.isKnownDatanode(datanodeID)) {
                throw new SCMException("Node already exists in the map", SCMException.ResultCodes.DUPLICATE_DATANODE);
            }
            this.scmNodeStorageReportMap.putIfAbsent(datanodeID, report);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateDatanodeMap(UUID datanodeID, Set<StorageLocationReport> report) throws SCMException {
        Preconditions.checkNotNull((Object)datanodeID);
        Preconditions.checkNotNull(report);
        Preconditions.checkState((!report.isEmpty() ? 1 : 0) != 0);
        Map<UUID, Set<StorageLocationReport>> map = this.scmNodeStorageReportMap;
        synchronized (map) {
            if (!this.scmNodeStorageReportMap.containsKey(datanodeID)) {
                throw new SCMException("No such datanode", SCMException.ResultCodes.NO_SUCH_DATANODE);
            }
            this.scmNodeStorageReportMap.put(datanodeID, report);
        }
    }

    public StorageReportResult processNodeReport(UUID datanodeID, StorageContainerDatanodeProtocolProtos.NodeReportProto nodeReport) throws IOException {
        Preconditions.checkNotNull((Object)datanodeID);
        Preconditions.checkNotNull((Object)nodeReport);
        long totalCapacity = 0L;
        long totalRemaining = 0L;
        long totalScmUsed = 0L;
        HashSet<StorageLocationReport> storagReportSet = new HashSet<StorageLocationReport>();
        HashSet<StorageLocationReport> fullVolumeSet = new HashSet<StorageLocationReport>();
        HashSet<StorageLocationReport> failedVolumeSet = new HashSet<StorageLocationReport>();
        List storageReports = nodeReport.getStorageReportList();
        for (StorageContainerDatanodeProtocolProtos.StorageReportProto report : storageReports) {
            StorageLocationReport storageReport = StorageLocationReport.getFromProtobuf((StorageContainerDatanodeProtocolProtos.StorageReportProto)report);
            storagReportSet.add(storageReport);
            if (report.hasFailed() && report.getFailed()) {
                failedVolumeSet.add(storageReport);
            } else if (this.isThresholdReached(UtilizationThreshold.CRITICAL, this.getScmUsedratio(report.getScmUsed(), report.getCapacity()))) {
                fullVolumeSet.add(storageReport);
            }
            totalCapacity += report.getCapacity();
            totalRemaining += report.getRemaining();
            totalScmUsed += report.getScmUsed();
        }
        if (!this.isKnownDatanode(datanodeID)) {
            this.insertNewDatanode(datanodeID, storagReportSet);
        } else {
            this.updateDatanodeMap(datanodeID, storagReportSet);
        }
        if (this.isThresholdReached(UtilizationThreshold.CRITICAL, this.getScmUsedratio(totalScmUsed, totalCapacity))) {
            LOG.warn("Datanode {} is out of storage space. Capacity: {}, Used: {}", new Object[]{datanodeID, totalCapacity, totalScmUsed});
            return StorageReportResult.ReportResultBuilder.newBuilder().setStatus(ReportStatus.DATANODE_OUT_OF_SPACE).setFullVolumeSet(fullVolumeSet).setFailedVolumeSet(failedVolumeSet).build();
        }
        if (this.isThresholdReached(UtilizationThreshold.WARN, this.getScmUsedratio(totalScmUsed, totalCapacity))) {
            LOG.warn("Datanode {} is low on storage space. Capacity: {}, Used: {}", new Object[]{datanodeID, totalCapacity, totalScmUsed});
        }
        if (failedVolumeSet.isEmpty() && !fullVolumeSet.isEmpty()) {
            return StorageReportResult.ReportResultBuilder.newBuilder().setStatus(ReportStatus.STORAGE_OUT_OF_SPACE).setFullVolumeSet(fullVolumeSet).build();
        }
        if (!failedVolumeSet.isEmpty() && fullVolumeSet.isEmpty()) {
            return StorageReportResult.ReportResultBuilder.newBuilder().setStatus(ReportStatus.FAILED_STORAGE).setFailedVolumeSet(failedVolumeSet).build();
        }
        if (!failedVolumeSet.isEmpty() && !fullVolumeSet.isEmpty()) {
            return StorageReportResult.ReportResultBuilder.newBuilder().setStatus(ReportStatus.FAILED_AND_OUT_OF_SPACE_STORAGE).setFailedVolumeSet(failedVolumeSet).setFullVolumeSet(fullVolumeSet).build();
        }
        return StorageReportResult.ReportResultBuilder.newBuilder().setStatus(ReportStatus.ALL_IS_WELL).build();
    }

    private boolean isThresholdReached(UtilizationThreshold threshold, double scmUsedratio) {
        switch (threshold.ordinal()) {
            case 0: {
                return scmUsedratio < this.warningUtilizationThreshold;
            }
            case 1: {
                return scmUsedratio >= this.warningUtilizationThreshold && scmUsedratio < this.criticalUtilizationThreshold;
            }
            case 2: {
                return scmUsedratio >= this.criticalUtilizationThreshold;
            }
        }
        throw new RuntimeException("Unknown UtilizationThreshold value");
    }

    @Override
    public long getCapacity(UUID dnId) {
        long capacity = 0L;
        Set<StorageLocationReport> reportSet = this.scmNodeStorageReportMap.get(dnId);
        for (StorageLocationReport report : reportSet) {
            capacity += report.getCapacity();
        }
        return capacity;
    }

    @Override
    public long getRemainingSpace(UUID dnId) {
        long remaining = 0L;
        Set<StorageLocationReport> reportSet = this.scmNodeStorageReportMap.get(dnId);
        for (StorageLocationReport report : reportSet) {
            remaining += report.getRemaining();
        }
        return remaining;
    }

    @Override
    public long getUsedSpace(UUID dnId) {
        long scmUsed = 0L;
        Set<StorageLocationReport> reportSet = this.scmNodeStorageReportMap.get(dnId);
        for (StorageLocationReport report : reportSet) {
            scmUsed += report.getScmUsed();
        }
        return scmUsed;
    }

    @Override
    public long getTotalCapacity() {
        long capacity = 0L;
        Set<UUID> dnIdSet = this.scmNodeStorageReportMap.keySet();
        for (UUID id : dnIdSet) {
            capacity += this.getCapacity(id);
        }
        return capacity;
    }

    @Override
    public long getTotalSpaceUsed() {
        long scmUsed = 0L;
        Set<UUID> dnIdSet = this.scmNodeStorageReportMap.keySet();
        for (UUID id : dnIdSet) {
            scmUsed += this.getUsedSpace(id);
        }
        return scmUsed;
    }

    @Override
    public long getTotalFreeSpace() {
        long remaining = 0L;
        Set<UUID> dnIdSet = this.scmNodeStorageReportMap.keySet();
        for (UUID id : dnIdSet) {
            remaining += this.getRemainingSpace(id);
        }
        return remaining;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void removeDatanode(UUID datanodeID) throws SCMException {
        Preconditions.checkNotNull((Object)datanodeID);
        Map<UUID, Set<StorageLocationReport>> map = this.scmNodeStorageReportMap;
        synchronized (map) {
            if (!this.scmNodeStorageReportMap.containsKey(datanodeID)) {
                throw new SCMException("No such datanode", SCMException.ResultCodes.NO_SUCH_DATANODE);
            }
            this.scmNodeStorageReportMap.remove(datanodeID);
        }
    }

    @Override
    public Set<StorageLocationReport> getStorageVolumes(UUID datanodeID) {
        return this.scmNodeStorageReportMap.get(datanodeID);
    }

    private double truncateDecimals(double value) {
        int multiplier = 10000;
        return (double)((long)(value * 10000.0)) / 10000.0;
    }

    public double getScmUsedratio(long scmUsed, long capacity) {
        double scmUsedRatio = this.truncateDecimals((double)scmUsed / (double)capacity);
        return scmUsedRatio;
    }

    public static enum UtilizationThreshold {
        NORMAL,
        WARN,
        CRITICAL;

    }

    public static enum ReportStatus {
        ALL_IS_WELL,
        DATANODE_OUT_OF_SPACE,
        STORAGE_OUT_OF_SPACE,
        FAILED_STORAGE,
        FAILED_AND_OUT_OF_SPACE_STORAGE;

    }
}

